import type { Metadata } from "next"
import { Suspense } from "react"
import { EarningsClient } from "@/components/admin/earnings/earnings-client"
import { earningsDataStore } from "@/lib/data/earnings"
import { Skeleton } from "@/components/ui/skeleton"

export const metadata: Metadata = {
  title: "Hotel Earnings | Teertham Admin",
  description:
    "View and manage all hotel partner earnings on Teertham platform. Track total revenue, monthly earnings, wallet balance, and withdrawal history for hotels.",
  openGraph: {
    title: "Hotel Earnings | Teertham Admin",
    description: "Comprehensive hotel earnings dashboard with detailed revenue reports and partner tracking.",
    type: "website",
    siteName: "Teertham",
  },
  twitter: {
    card: "summary",
    title: "Hotel Earnings | Teertham Admin",
    description: "Comprehensive hotel earnings dashboard with detailed revenue reports and partner tracking.",
  },
  robots: {
    index: false,
    follow: false,
  },
}

function EarningsTableSkeleton() {
  return (
    <div className="space-y-6">
      {/* Summary cards skeleton */}
      <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-4">
        {Array.from({ length: 4 }).map((_, i) => (
          <Skeleton key={i} className="h-24 w-full" />
        ))}
      </div>
      {/* Search skeleton */}
      <div className="flex items-center justify-between">
        <Skeleton className="h-10 w-72" />
        <Skeleton className="h-10 w-24" />
      </div>
      {/* Table skeleton */}
      <Skeleton className="h-96 w-full" />
    </div>
  )
}

export default function HotelEarningPage() {
  const hotelEarnings = earningsDataStore.getHotelEarnings()
  const summary = earningsDataStore.getTotalEarningsByType("hotel")

  const jsonLd = {
    "@context": "https://schema.org",
    "@type": "WebPage",
    name: "Hotel Earnings Dashboard",
    description: "Administrative dashboard for managing hotel partner earnings and revenue on Teertham platform.",
    isPartOf: {
      "@type": "WebSite",
      name: "Teertham Admin",
    },
  }

  return (
    <>
      <script type="application/ld+json" dangerouslySetInnerHTML={{ __html: JSON.stringify(jsonLd) }} />
      <main className="space-y-6">
        <header>
          <h1 className="text-3xl font-bold tracking-tight">Hotel Earnings</h1>
          <p className="text-muted-foreground mt-2">View earnings reports and revenue details for all hotel partners</p>
        </header>

        <Suspense fallback={<EarningsTableSkeleton />}>
          <EarningsClient data={hotelEarnings} summary={summary} userType="hotel" />
        </Suspense>
      </main>
    </>
  )
}
