import type { Metadata } from "next"
import { Suspense } from "react"
import { RouteForm } from "@/components/admin/routes/route-form"
import { Skeleton } from "@/components/ui/skeleton"

export const metadata: Metadata = {
  title: "Add New Route | Teertham Admin",
  description: "Create a new vehicle route with origin, destination, and vehicle pricing configuration.",
  robots: { index: false, follow: false },
  openGraph: {
    title: "Add New Route | Teertham Admin",
    description: "Create a new vehicle route with origin, destination, and vehicle pricing.",
    type: "website",
  },
}

function FormLoadingSkeleton() {
  return (
    <div className="space-y-6">
      <div className="flex items-center gap-4">
        <Skeleton className="h-10 w-10 rounded-full" />
        <div className="space-y-2">
          <Skeleton className="h-8 w-48" />
          <Skeleton className="h-4 w-64" />
        </div>
      </div>
      <Skeleton className="h-48 w-full" />
      <Skeleton className="h-32 w-full" />
      <Skeleton className="h-64 w-full" />
    </div>
  )
}

interface PageProps {
  searchParams: Promise<{ id?: string }>
}

export default async function AddRoutePage({ searchParams }: PageProps) {
  const params = await searchParams
  const routeId = params.id

  return (
    <main className="flex-1 p-4 sm:p-6 lg:p-8">
      <Suspense fallback={<FormLoadingSkeleton />}>
        <RouteForm routeId={routeId} />
      </Suspense>

      <script
        type="application/ld+json"
        dangerouslySetInnerHTML={{
          __html: JSON.stringify({
            "@context": "https://schema.org",
            "@type": "WebPage",
            name: routeId ? "Edit Route" : "Add New Route",
            description: "Configure vehicle route with pricing.",
            isPartOf: {
              "@type": "WebSite",
              name: "Teertham Admin",
            },
          }),
        }}
      />
    </main>
  )
}
