"use client"

import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from "@/components/ui/dialog"
import { Badge } from "@/components/ui/badge"
import { Separator } from "@/components/ui/separator"
import { ScrollArea } from "@/components/ui/scroll-area"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { User, Mail, Phone, MapPin, Calendar, IndianRupee, FileText, Clock, AlertTriangle, XCircle } from "lucide-react"
import type { Agent, AgentStatus } from "@/lib/data/agents"

interface AgentDetailsDialogProps {
  agent: Agent | null
  open: boolean
  onOpenChange: (open: boolean) => void
}

const statusConfig: Record<
  AgentStatus,
  { label: string; variant: "default" | "secondary" | "destructive" | "outline" }
> = {
  pending: { label: "Pending Review", variant: "secondary" },
  approved: { label: "Approved", variant: "default" },
  rejected: { label: "Rejected", variant: "outline" },
  blocked: { label: "Blocked", variant: "destructive" },
}

export function AgentDetailsDialog({ agent, open, onOpenChange }: AgentDetailsDialogProps) {
  if (!agent) return null

  const status = statusConfig[agent.status]

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-IN", {
      style: "currency",
      currency: "INR",
      maximumFractionDigits: 0,
    }).format(amount)
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-IN", {
      year: "numeric",
      month: "long",
      day: "numeric",
    })
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-2xl max-h-[90vh]">
        <DialogHeader>
          <div className="flex items-center justify-between">
            <div>
              <DialogTitle className="text-xl">
                {agent.firstName} {agent.lastName}
              </DialogTitle>
              <DialogDescription>Agent Details</DialogDescription>
            </div>
            <Badge variant={status.variant}>{status.label}</Badge>
          </div>
        </DialogHeader>

        <ScrollArea className="max-h-[calc(90vh-120px)] pr-4">
          <div className="space-y-6">
            {/* Contact Information */}
            <section aria-labelledby="contact-heading">
              <h3 id="contact-heading" className="text-sm font-semibold text-muted-foreground mb-3">
                Contact Information
              </h3>
              <div className="grid gap-3">
                <div className="flex items-center gap-3">
                  <Mail className="h-4 w-4 text-muted-foreground" aria-hidden="true" />
                  <span>{agent.email}</span>
                </div>
                <div className="flex items-center gap-3">
                  <Phone className="h-4 w-4 text-muted-foreground" aria-hidden="true" />
                  <span>{agent.phone}</span>
                </div>
                <div className="flex items-start gap-3">
                  <MapPin className="h-4 w-4 text-muted-foreground mt-1" aria-hidden="true" />
                  <span>{agent.officeAddress}</span>
                </div>
              </div>
            </section>

            <Separator />

            {/* Business Documents */}
            <section aria-labelledby="documents-heading">
              <h3 id="documents-heading" className="text-sm font-semibold text-muted-foreground mb-3">
                Business Documents
              </h3>
              <div className="grid gap-3 sm:grid-cols-2">
                <div className="flex items-center gap-3 rounded-lg border p-3">
                  <FileText className="h-5 w-5 text-muted-foreground" aria-hidden="true" />
                  <div>
                    <p className="text-sm font-medium">GST Certificate</p>
                    <p className="text-xs text-muted-foreground">{agent.gstCertificate}</p>
                  </div>
                </div>
                <div className="flex items-center gap-3 rounded-lg border p-3">
                  <FileText className="h-5 w-5 text-muted-foreground" aria-hidden="true" />
                  <div>
                    <p className="text-sm font-medium">MSME Certificate</p>
                    <p className="text-xs text-muted-foreground">{agent.msmeCertificate}</p>
                  </div>
                </div>
              </div>
            </section>

            <Separator />

            {/* Statistics */}
            <section aria-labelledby="stats-heading">
              <h3 id="stats-heading" className="text-sm font-semibold text-muted-foreground mb-3">
                Performance Statistics
              </h3>
              <div className="grid gap-4 sm:grid-cols-2">
                <div className="flex items-center gap-3 rounded-lg bg-muted/50 p-4">
                  <User className="h-5 w-5 text-primary" aria-hidden="true" />
                  <div>
                    <p className="text-2xl font-bold">{agent.totalBookings}</p>
                    <p className="text-sm text-muted-foreground">Total Bookings</p>
                  </div>
                </div>
                <div className="flex items-center gap-3 rounded-lg bg-muted/50 p-4">
                  <IndianRupee className="h-5 w-5 text-primary" aria-hidden="true" />
                  <div>
                    <p className="text-2xl font-bold">{formatCurrency(agent.totalCommission)}</p>
                    <p className="text-sm text-muted-foreground">Total Commission</p>
                  </div>
                </div>
              </div>
            </section>

            <Separator />

            {/* Timeline */}
            <section aria-labelledby="timeline-heading">
              <h3 id="timeline-heading" className="text-sm font-semibold text-muted-foreground mb-3">
                Timeline
              </h3>
              <div className="grid gap-3 sm:grid-cols-2">
                <div className="flex items-center gap-3">
                  <Calendar className="h-4 w-4 text-muted-foreground" aria-hidden="true" />
                  <div>
                    <p className="text-sm font-medium">Joined</p>
                    <p className="text-xs text-muted-foreground">{formatDate(agent.joinedAt)}</p>
                  </div>
                </div>
                <div className="flex items-center gap-3">
                  <Clock className="h-4 w-4 text-muted-foreground" aria-hidden="true" />
                  <div>
                    <p className="text-sm font-medium">Last Active</p>
                    <p className="text-xs text-muted-foreground">{formatDate(agent.lastActive)}</p>
                  </div>
                </div>
                {agent.approvedAt && (
                  <div className="flex items-center gap-3">
                    <Calendar className="h-4 w-4 text-green-600" aria-hidden="true" />
                    <div>
                      <p className="text-sm font-medium">Approved On</p>
                      <p className="text-xs text-muted-foreground">{formatDate(agent.approvedAt)}</p>
                    </div>
                  </div>
                )}
              </div>
            </section>

            {/* Block/Rejection Reason */}
            {agent.status === "blocked" && agent.blockReason && (
              <>
                <Separator />
                <section aria-labelledby="block-reason-heading">
                  <div className="flex items-start gap-3 rounded-lg border border-destructive/50 bg-destructive/10 p-4">
                    <AlertTriangle className="h-5 w-5 text-destructive mt-0.5" aria-hidden="true" />
                    <div>
                      <h3 id="block-reason-heading" className="text-sm font-semibold text-destructive">
                        Block Reason
                      </h3>
                      <p className="text-sm mt-1">{agent.blockReason}</p>
                      {agent.blockedAt && (
                        <p className="text-xs text-muted-foreground mt-2">Blocked on {formatDate(agent.blockedAt)}</p>
                      )}
                    </div>
                  </div>
                </section>
              </>
            )}

            {agent.status === "rejected" && agent.rejectionReason && (
              <>
                <Separator />
                <section aria-labelledby="rejection-reason-heading">
                  <div className="flex items-start gap-3 rounded-lg border border-orange-500/50 bg-orange-500/10 p-4">
                    <XCircle className="h-5 w-5 text-orange-600 mt-0.5" aria-hidden="true" />
                    <div>
                      <h3 id="rejection-reason-heading" className="text-sm font-semibold text-orange-600">
                        Rejection Reason
                      </h3>
                      <p className="text-sm mt-1">{agent.rejectionReason}</p>
                      {agent.rejectedAt && (
                        <p className="text-xs text-muted-foreground mt-2">Rejected on {formatDate(agent.rejectedAt)}</p>
                      )}
                    </div>
                  </div>
                </section>
              </>
            )}

            {/* Booking History */}
            {agent.bookings.length > 0 && (
              <>
                <Separator />
                <section aria-labelledby="bookings-heading">
                  <h3 id="bookings-heading" className="text-sm font-semibold text-muted-foreground mb-3">
                    Recent Bookings
                  </h3>
                  <div className="rounded-lg border">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>Customer</TableHead>
                          <TableHead className="hidden sm:table-cell">Package</TableHead>
                          <TableHead>Commission</TableHead>
                          <TableHead>Status</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {agent.bookings.map((booking) => (
                          <TableRow key={booking.id}>
                            <TableCell className="font-medium">{booking.customerName}</TableCell>
                            <TableCell className="hidden sm:table-cell">{booking.packageName}</TableCell>
                            <TableCell>{formatCurrency(booking.commission)}</TableCell>
                            <TableCell>
                              <Badge
                                variant={
                                  booking.status === "completed"
                                    ? "default"
                                    : booking.status === "upcoming"
                                      ? "secondary"
                                      : "destructive"
                                }
                              >
                                {booking.status}
                              </Badge>
                            </TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </div>
                </section>
              </>
            )}
          </div>
        </ScrollArea>
      </DialogContent>
    </Dialog>
  )
}
