"use client"

import { useState, useEffect, useCallback } from "react"
import { useRouter } from "next/navigation"
import Image from "next/image"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { ArrowLeft, Save, MapPin, Car, Plus, Trash2, IndianRupee } from "lucide-react"
import { routeDataStore } from "@/lib/data/routes"
import { locationDataStore, type City } from "@/lib/data/locations"
import { vehicleDataStore, type Vehicle } from "@/lib/data/vehicles"
import { useToastEnhanced } from "@/hooks/use-toast-enhanced"

interface RouteFormProps {
  routeId?: string
}

interface RouteVehicleFormData {
  tempId: string
  vehicleId: string
  vehicleName: string
  vehicleCategory: string
  vehiclePhoto: string
  basePerKmRate: number
  routePerKmRate: number
}

export function RouteForm({ routeId }: RouteFormProps) {
  const router = useRouter()
  const toast = useToastEnhanced()
  const isEditing = Boolean(routeId)

  const [cities, setCities] = useState<City[]>([])
  const [vehicles, setVehicles] = useState<Vehicle[]>([])
  const [isSubmitting, setIsSubmitting] = useState(false)

  // Form state
  const [originCityId, setOriginCityId] = useState("")
  const [destinationCityId, setDestinationCityId] = useState("")
  const [routeVehicles, setRouteVehicles] = useState<RouteVehicleFormData[]>([])

  // Vehicle selection state
  const [selectedVehicleId, setSelectedVehicleId] = useState("")

  // Load data on mount
  useEffect(() => {
    const activeCities = locationDataStore.getActiveCities()
    const activeVehicles = vehicleDataStore.getActiveVehicles()
    setCities(activeCities)
    setVehicles(activeVehicles)
  }, [])

  // Load route data if editing
  useEffect(() => {
    if (routeId) {
      const route = routeDataStore.getRouteById(routeId)
      if (route) {
        setOriginCityId(route.originCityId)
        setDestinationCityId(route.destinationCityId)
        setRouteVehicles(
          route.vehicles.map((v) => ({
            tempId: v.id,
            vehicleId: v.vehicleId,
            vehicleName: v.vehicleName,
            vehicleCategory: v.vehicleCategory,
            vehiclePhoto: v.vehiclePhoto,
            basePerKmRate: v.basePerKmRate,
            routePerKmRate: v.routePerKmRate,
          })),
        )
      }
    }
  }, [routeId])

  const getOriginCityName = () => {
    const city = cities.find((c) => c.id === originCityId)
    return city?.name || ""
  }

  const getDestinationCityName = () => {
    const city = cities.find((c) => c.id === destinationCityId)
    return city?.name || ""
  }

  const handleAddVehicle = useCallback(() => {
    if (!selectedVehicleId) {
      toast.error({
        title: "Select Vehicle",
        description: "Please select a vehicle to add.",
      })
      return
    }

    // Check if vehicle already added
    if (routeVehicles.some((v) => v.vehicleId === selectedVehicleId)) {
      toast.error({
        title: "Vehicle Already Added",
        description: "This vehicle is already added to this route.",
      })
      return
    }

    const vehicle = vehicles.find((v) => v.id === selectedVehicleId)
    if (!vehicle) return

    const newVehicle: RouteVehicleFormData = {
      tempId: `temp_${Date.now()}`,
      vehicleId: vehicle.id,
      vehicleName: vehicle.name,
      vehicleCategory: vehicle.categoryName,
      vehiclePhoto: vehicle.photo,
      basePerKmRate: vehicle.perKmPrice,
      routePerKmRate: vehicle.perKmPrice, // Default to base rate
    }

    setRouteVehicles((prev) => [...prev, newVehicle])
    setSelectedVehicleId("")
    toast.success({
      title: "Vehicle Added",
      description: `${vehicle.name} has been added to the route.`,
    })
  }, [selectedVehicleId, routeVehicles, vehicles, toast])

  const handleRemoveVehicle = useCallback(
    (tempId: string) => {
      const vehicle = routeVehicles.find((v) => v.tempId === tempId)
      setRouteVehicles((prev) => prev.filter((v) => v.tempId !== tempId))
      if (vehicle) {
        toast.success({
          title: "Vehicle Removed",
          description: `${vehicle.vehicleName} has been removed from the route.`,
        })
      }
    },
    [routeVehicles, toast],
  )

  const handleUpdateVehicleRate = useCallback((tempId: string, rate: number) => {
    setRouteVehicles((prev) => prev.map((v) => (v.tempId === tempId ? { ...v, routePerKmRate: rate } : v)))
  }, [])

  const handleSubmit = async () => {
    // Validation
    if (!originCityId) {
      toast.error({
        title: "Origin Required",
        description: "Please select an origin city.",
      })
      return
    }

    if (!destinationCityId) {
      toast.error({
        title: "Destination Required",
        description: "Please select a destination city.",
      })
      return
    }

    if (originCityId === destinationCityId) {
      toast.error({
        title: "Invalid Route",
        description: "Origin and destination cannot be the same.",
      })
      return
    }

    if (routeVehicles.length === 0) {
      toast.error({
        title: "Vehicles Required",
        description: "Please add at least one vehicle to the route.",
      })
      return
    }

    setIsSubmitting(true)

    // Simulate network delay
    await new Promise((resolve) => setTimeout(resolve, 500))

    const routeData = {
      origin: getOriginCityName(),
      originCityId,
      destination: getDestinationCityName(),
      destinationCityId,
      vehicles: routeVehicles.map((v) => ({
        vehicleId: v.vehicleId,
        vehicleName: v.vehicleName,
        vehicleCategory: v.vehicleCategory,
        vehiclePhoto: v.vehiclePhoto,
        basePerKmRate: v.basePerKmRate,
        routePerKmRate: v.routePerKmRate,
      })),
    }

    if (isEditing && routeId) {
      const updated = routeDataStore.updateRoute(routeId, routeData)
      if (updated) {
        toast.success({
          title: "Route Updated",
          description: `Route from ${routeData.origin} to ${routeData.destination} has been updated.`,
        })
        router.push("/admin/routes-listing")
      } else {
        toast.error({
          title: "Update Failed",
          description: "This route already exists or invalid data provided.",
        })
      }
    } else {
      const created = routeDataStore.addRoute(routeData)
      if (created) {
        toast.success({
          title: "Route Created",
          description: `Route from ${routeData.origin} to ${routeData.destination} has been created.`,
        })
        router.push("/admin/routes-listing")
      } else {
        toast.error({
          title: "Creation Failed",
          description: "This route already exists.",
        })
      }
    }

    setIsSubmitting(false)
  }

  // Filter out already added vehicles from selection
  const availableVehicles = vehicles.filter((v) => !routeVehicles.some((rv) => rv.vehicleId === v.id))

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center gap-4">
        <Button variant="ghost" size="icon" onClick={() => router.back()} aria-label="Go back">
          <ArrowLeft className="h-5 w-5" />
        </Button>
        <div>
          <h1 className="text-2xl font-bold tracking-tight">{isEditing ? "Edit Route" : "Add New Route"}</h1>
          <p className="text-muted-foreground">
            {isEditing ? "Update route details and vehicle pricing" : "Create a new route with vehicle assignments"}
          </p>
        </div>
      </div>

      {/* Route Details */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <MapPin className="h-5 w-5 text-primary" aria-hidden="true" />
            Route Details
          </CardTitle>
          <CardDescription>Select origin and destination cities</CardDescription>
        </CardHeader>
        <CardContent className="grid gap-4 sm:grid-cols-2">
          <div className="space-y-2">
            <Label htmlFor="origin">Origin City *</Label>
            <Select value={originCityId} onValueChange={setOriginCityId}>
              <SelectTrigger id="origin">
                <SelectValue placeholder="Select origin city" />
              </SelectTrigger>
              <SelectContent>
                {cities.map((city) => (
                  <SelectItem key={city.id} value={city.id} disabled={city.id === destinationCityId}>
                    {city.name} ({city.stateName})
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          <div className="space-y-2">
            <Label htmlFor="destination">Destination City *</Label>
            <Select value={destinationCityId} onValueChange={setDestinationCityId}>
              <SelectTrigger id="destination">
                <SelectValue placeholder="Select destination city" />
              </SelectTrigger>
              <SelectContent>
                {cities.map((city) => (
                  <SelectItem key={city.id} value={city.id} disabled={city.id === originCityId}>
                    {city.name} ({city.stateName})
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </CardContent>
      </Card>

      {/* Add Vehicle */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Car className="h-5 w-5 text-primary" aria-hidden="true" />
            Add Vehicle
          </CardTitle>
          <CardDescription>Select a vehicle to add to this route</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="flex flex-col gap-4 sm:flex-row sm:items-end">
            <div className="flex-1 space-y-2">
              <Label htmlFor="vehicle-select">Select Vehicle</Label>
              <Select value={selectedVehicleId} onValueChange={setSelectedVehicleId}>
                <SelectTrigger id="vehicle-select">
                  <SelectValue placeholder="Choose a vehicle to add" />
                </SelectTrigger>
                <SelectContent>
                  {availableVehicles.length === 0 ? (
                    <SelectItem value="none" disabled>
                      No vehicles available
                    </SelectItem>
                  ) : (
                    availableVehicles.map((vehicle) => (
                      <SelectItem key={vehicle.id} value={vehicle.id}>
                        {vehicle.name} ({vehicle.categoryName}) - ₹{vehicle.perKmPrice}/km
                      </SelectItem>
                    ))
                  )}
                </SelectContent>
              </Select>
            </div>
            <Button onClick={handleAddVehicle} disabled={!selectedVehicleId}>
              <Plus className="mr-2 h-4 w-4" aria-hidden="true" />
              Add Vehicle
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Added Vehicles */}
      {routeVehicles.length > 0 && (
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <IndianRupee className="h-5 w-5 text-primary" aria-hidden="true" />
              Route Vehicles & Pricing
            </CardTitle>
            <CardDescription>
              {routeVehicles.length} vehicle(s) added. Set route-specific pricing for each vehicle.
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {routeVehicles.map((vehicle) => (
                <div
                  key={vehicle.tempId}
                  className="flex flex-col gap-4 rounded-lg border p-4 sm:flex-row sm:items-center"
                >
                  <div className="flex items-center gap-4 flex-1">
                    <div className="relative h-16 w-24 overflow-hidden rounded-md border bg-muted shrink-0">
                      <Image
                        src={vehicle.vehiclePhoto || "/placeholder.svg"}
                        alt={vehicle.vehicleName}
                        fill
                        className="object-cover"
                        sizes="96px"
                      />
                    </div>
                    <div className="min-w-0 flex-1">
                      <h4 className="font-medium truncate">{vehicle.vehicleName}</h4>
                      <p className="text-sm text-muted-foreground">{vehicle.vehicleCategory}</p>
                      <p className="text-xs text-muted-foreground">Base Rate: ₹{vehicle.basePerKmRate}/km</p>
                    </div>
                  </div>

                  <div className="flex items-center gap-4">
                    <div className="space-y-1">
                      <Label htmlFor={`rate-${vehicle.tempId}`} className="text-xs">
                        Route Total Rate
                      </Label>
                      <Input
                        id={`rate-${vehicle.tempId}`}
                        type="number"
                        min={1}
                        value={vehicle.routePerKmRate}
                        onChange={(e) => handleUpdateVehicleRate(vehicle.tempId, Number(e.target.value) || 1)}
                        className="w-24"
                      />
                    </div>
                    <Button
                      variant="ghost"
                      size="icon"
                      onClick={() => handleRemoveVehicle(vehicle.tempId)}
                      className="text-destructive hover:text-destructive shrink-0"
                      aria-label={`Remove ${vehicle.vehicleName}`}
                    >
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}

      {/* Actions */}
      <div className="flex items-center justify-end gap-4">
        <Button type="button" variant="outline" onClick={() => router.back()} disabled={isSubmitting}>
          Cancel
        </Button>
        <Button onClick={handleSubmit} disabled={isSubmitting}>
          <Save className="mr-2 h-4 w-4" aria-hidden="true" />
          {isSubmitting ? "Saving..." : isEditing ? "Update Route" : "Create Route"}
        </Button>
      </div>
    </div>
  )
}
