"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Copy, Check } from "lucide-react"
import { defaultCredentials } from "@/lib/data/users"

interface CredentialItemProps {
  title: string
  email: string
  password: string
}

function CredentialItem({ title, email, password }: CredentialItemProps) {
  const [copiedEmail, setCopiedEmail] = useState(false)
  const [copiedPassword, setCopiedPassword] = useState(false)

  const copyToClipboard = async (text: string, type: "email" | "password") => {
    try {
      await navigator.clipboard.writeText(text)
      if (type === "email") {
        setCopiedEmail(true)
        setTimeout(() => setCopiedEmail(false), 2000)
      } else {
        setCopiedPassword(true)
        setTimeout(() => setCopiedPassword(false), 2000)
      }
    } catch (err) {
      console.error("Failed to copy:", err)
    }
  }

  return (
    <div className="bg-background rounded-lg p-3 space-y-2 border border-border/50">
      <p className="font-medium text-primary text-sm">{title}</p>
      <div className="flex items-center justify-between gap-2">
        <p className="text-xs text-muted-foreground truncate">
          <span className="font-medium">Email:</span> {email}
        </p>
        <Button
          type="button"
          variant="ghost"
          size="sm"
          className="h-6 w-6 p-0 hover:bg-primary/10"
          onClick={() => copyToClipboard(email, "email")}
          aria-label={`Copy email ${email}`}
        >
          {copiedEmail ? (
            <Check className="h-3 w-3 text-green-500" aria-hidden="true" />
          ) : (
            <Copy className="h-3 w-3" aria-hidden="true" />
          )}
        </Button>
      </div>
      <div className="flex items-center justify-between gap-2">
        <p className="text-xs text-muted-foreground truncate">
          <span className="font-medium">Password:</span> {password}
        </p>
        <Button
          type="button"
          variant="ghost"
          size="sm"
          className="h-6 w-6 p-0 hover:bg-primary/10"
          onClick={() => copyToClipboard(password, "password")}
          aria-label={`Copy password for ${title}`}
        >
          {copiedPassword ? (
            <Check className="h-3 w-3 text-green-500" aria-hidden="true" />
          ) : (
            <Copy className="h-3 w-3" aria-hidden="true" />
          )}
        </Button>
      </div>
    </div>
  )
}

export function DemoCredentials() {
  return (
    <section className="bg-muted/50 border rounded-xl p-4 space-y-3">
      <h2 className="text-sm font-semibold text-center">Demo Credentials</h2>
      <div className="grid gap-2">
        <CredentialItem
          title="Admin Account"
          email={defaultCredentials.admin.email}
          password={defaultCredentials.admin.password}
        />
        <CredentialItem
          title="Customer Account"
          email={defaultCredentials.customer.email}
          password={defaultCredentials.customer.password}
        />
        <CredentialItem
          title="Agent Account"
          email={defaultCredentials.agent.email}
          password={defaultCredentials.agent.password}
        />
        <CredentialItem
          title="Hotel Account"
          email={defaultCredentials.hotel.email}
          password={defaultCredentials.hotel.password}
        />
      </div>
    </section>
  )
}
