import { hotelPartnerDataStore } from "./hotel-partners"

export type BookingStatus = "booked" | "ongoing" | "completed" | "cancelled"

export interface HotelBooking {
  id: string
  hotelId: string
  hotelName: string
  hotelOwnerName: string
  hotelEmail: string
  hotelPhone: string
  customerId: string
  customerName: string
  customerEmail: string
  customerPhone: string
  stateId: string
  stateName: string
  cityId: string
  cityName: string
  roomSelected: string
  totalMembers: number
  checkInDate: string
  checkOutDate: string
  totalPrice: number
  commission: number
  status: BookingStatus
  createdAt: string
  updatedAt: string
}

// In-memory hotel booking data store
class HotelBookingDataStore {
  private bookings: Map<string, HotelBooking> = new Map()
  private idCounter = 0

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const partners = hotelPartnerDataStore.getAllPartners()
    const customerData = [
      { name: "Amit Kumar", email: "amit.kumar@email.com", phone: "+91 9876501001" },
      { name: "Priya Sharma", email: "priya.sharma@email.com", phone: "+91 9876501002" },
      { name: "Rajesh Patel", email: "rajesh.patel@email.com", phone: "+91 9876501003" },
      { name: "Sunita Verma", email: "sunita.verma@email.com", phone: "+91 9876501004" },
      { name: "Vikram Singh", email: "vikram.singh@email.com", phone: "+91 9876501005" },
      { name: "Anita Reddy", email: "anita.reddy@email.com", phone: "+91 9876501006" },
      { name: "Suresh Gupta", email: "suresh.gupta@email.com", phone: "+91 9876501007" },
      { name: "Meena Iyer", email: "meena.iyer@email.com", phone: "+91 9876501008" },
      { name: "Arjun Nair", email: "arjun.nair@email.com", phone: "+91 9876501009" },
      { name: "Kavita Desai", email: "kavita.desai@email.com", phone: "+91 9876501010" },
      { name: "Rohit Menon", email: "rohit.menon@email.com", phone: "+91 9876501011" },
      { name: "Deepa Krishnan", email: "deepa.krishnan@email.com", phone: "+91 9876501012" },
      { name: "Sanjay Rao", email: "sanjay.rao@email.com", phone: "+91 9876501013" },
      { name: "Pooja Agarwal", email: "pooja.agarwal@email.com", phone: "+91 9876501014" },
      { name: "Manoj Pillai", email: "manoj.pillai@email.com", phone: "+91 9876501015" },
      { name: "Lakshmi Venkat", email: "lakshmi.venkat@email.com", phone: "+91 9876501016" },
      { name: "Ravi Choudhary", email: "ravi.choudhary@email.com", phone: "+91 9876501017" },
      { name: "Neha Joshi", email: "neha.joshi@email.com", phone: "+91 9876501018" },
      { name: "Kiran Murthy", email: "kiran.murthy@email.com", phone: "+91 9876501019" },
      { name: "Divya Saxena", email: "divya.saxena@email.com", phone: "+91 9876501020" },
    ]

    const roomTypes = ["Deluxe Room", "Standard Room", "Suite", "Family Room", "Executive Room"]

    const sampleBookings: Array<{
      hotelIndex: number
      customerIndex: number
      roomSelected: string
      totalMembers: number
      checkInDate: string
      checkOutDate: string
      totalPrice: number
      commission: number
      status: BookingStatus
    }> = [
      // Booked (upcoming)
      {
        hotelIndex: 0,
        customerIndex: 0,
        roomSelected: roomTypes[0],
        totalMembers: 4,
        checkInDate: "2024-02-15",
        checkOutDate: "2024-02-18",
        totalPrice: 12500,
        commission: 1250,
        status: "booked",
      },
      {
        hotelIndex: 1,
        customerIndex: 1,
        roomSelected: roomTypes[1],
        totalMembers: 2,
        checkInDate: "2024-02-20",
        checkOutDate: "2024-02-23",
        totalPrice: 8900,
        commission: 890,
        status: "booked",
      },
      {
        hotelIndex: 2,
        customerIndex: 2,
        roomSelected: roomTypes[2],
        totalMembers: 6,
        checkInDate: "2024-02-22",
        checkOutDate: "2024-02-25",
        totalPrice: 18750,
        commission: 1875,
        status: "booked",
      },
      {
        hotelIndex: 3,
        customerIndex: 3,
        roomSelected: roomTypes[3],
        totalMembers: 3,
        checkInDate: "2024-02-25",
        checkOutDate: "2024-02-28",
        totalPrice: 10200,
        commission: 1020,
        status: "booked",
      },
      {
        hotelIndex: 4,
        customerIndex: 4,
        roomSelected: roomTypes[4],
        totalMembers: 5,
        checkInDate: "2024-03-01",
        checkOutDate: "2024-03-04",
        totalPrice: 15600,
        commission: 1560,
        status: "booked",
      },

      // Ongoing (currently staying)
      {
        hotelIndex: 5,
        customerIndex: 5,
        roomSelected: roomTypes[0],
        totalMembers: 2,
        checkInDate: "2024-02-08",
        checkOutDate: "2024-02-11",
        totalPrice: 6800,
        commission: 680,
        status: "ongoing",
      },
      {
        hotelIndex: 6,
        customerIndex: 6,
        roomSelected: roomTypes[1],
        totalMembers: 4,
        checkInDate: "2024-02-07",
        checkOutDate: "2024-02-10",
        totalPrice: 13400,
        commission: 1340,
        status: "ongoing",
      },
      {
        hotelIndex: 7,
        customerIndex: 7,
        roomSelected: roomTypes[2],
        totalMembers: 3,
        checkInDate: "2024-02-06",
        checkOutDate: "2024-02-09",
        totalPrice: 11200,
        commission: 1120,
        status: "ongoing",
      },
      {
        hotelIndex: 0,
        customerIndex: 8,
        roomSelected: roomTypes[3],
        totalMembers: 5,
        checkInDate: "2024-02-05",
        checkOutDate: "2024-02-10",
        totalPrice: 19500,
        commission: 1950,
        status: "ongoing",
      },

      // Completed (past stays)
      {
        hotelIndex: 1,
        customerIndex: 9,
        roomSelected: roomTypes[0],
        totalMembers: 3,
        checkInDate: "2024-01-05",
        checkOutDate: "2024-01-08",
        totalPrice: 9500,
        commission: 950,
        status: "completed",
      },
      {
        hotelIndex: 2,
        customerIndex: 10,
        roomSelected: roomTypes[4],
        totalMembers: 5,
        checkInDate: "2024-01-10",
        checkOutDate: "2024-01-14",
        totalPrice: 17800,
        commission: 1780,
        status: "completed",
      },
      {
        hotelIndex: 3,
        customerIndex: 11,
        roomSelected: roomTypes[1],
        totalMembers: 2,
        checkInDate: "2024-01-15",
        checkOutDate: "2024-01-17",
        totalPrice: 6200,
        commission: 620,
        status: "completed",
      },
      {
        hotelIndex: 4,
        customerIndex: 12,
        roomSelected: roomTypes[2],
        totalMembers: 4,
        checkInDate: "2024-01-18",
        checkOutDate: "2024-01-21",
        totalPrice: 11900,
        commission: 1190,
        status: "completed",
      },
      {
        hotelIndex: 5,
        customerIndex: 13,
        roomSelected: roomTypes[3],
        totalMembers: 6,
        checkInDate: "2024-01-20",
        checkOutDate: "2024-01-24",
        totalPrice: 22400,
        commission: 2240,
        status: "completed",
      },

      // Cancelled
      {
        hotelIndex: 6,
        customerIndex: 14,
        roomSelected: roomTypes[0],
        totalMembers: 4,
        checkInDate: "2024-02-10",
        checkOutDate: "2024-02-13",
        totalPrice: 12000,
        commission: 1200,
        status: "cancelled",
      },
      {
        hotelIndex: 7,
        customerIndex: 15,
        roomSelected: roomTypes[1],
        totalMembers: 2,
        checkInDate: "2024-02-12",
        checkOutDate: "2024-02-14",
        totalPrice: 7500,
        commission: 750,
        status: "cancelled",
      },
      {
        hotelIndex: 0,
        customerIndex: 16,
        roomSelected: roomTypes[4],
        totalMembers: 3,
        checkInDate: "2024-02-14",
        checkOutDate: "2024-02-17",
        totalPrice: 10500,
        commission: 1050,
        status: "cancelled",
      },
    ]

    sampleBookings.forEach((booking, index) => {
      const partner = partners[booking.hotelIndex % partners.length]
      const customer = customerData[booking.customerIndex % customerData.length]
      if (partner && customer) {
        this.idCounter++
        const id = `booking_${String(this.idCounter).padStart(4, "0")}`
        const now = new Date().toISOString().split("T")[0]

        this.bookings.set(id, {
          id,
          hotelId: partner.id,
          hotelName: partner.hotelName,
          hotelOwnerName: partner.ownerName,
          hotelEmail: partner.email,
          hotelPhone: partner.phone,
          customerId: `customer_${String(booking.customerIndex + 1).padStart(3, "0")}`,
          customerName: customer.name,
          customerEmail: customer.email,
          customerPhone: customer.phone,
          stateId: partner.stateId,
          stateName: partner.stateName,
          cityId: partner.cityId,
          cityName: partner.cityName,
          roomSelected: booking.roomSelected,
          totalMembers: booking.totalMembers,
          checkInDate: booking.checkInDate,
          checkOutDate: booking.checkOutDate,
          totalPrice: booking.totalPrice,
          commission: booking.commission,
          status: booking.status,
          createdAt: now,
          updatedAt: now,
        })
      }
    })
  }

  getAllBookings(): HotelBooking[] {
    return Array.from(this.bookings.values()).sort(
      (a, b) => new Date(b.checkInDate).getTime() - new Date(a.checkInDate).getTime(),
    )
  }

  getBookingsByStatus(status: BookingStatus): HotelBooking[] {
    return this.getAllBookings().filter((b) => b.status === status)
  }

  getBookedBookings(): HotelBooking[] {
    return this.getBookingsByStatus("booked")
  }

  getOngoingBookings(): HotelBooking[] {
    return this.getBookingsByStatus("ongoing")
  }

  getCompletedBookings(): HotelBooking[] {
    return this.getBookingsByStatus("completed")
  }

  getCancelledBookings(): HotelBooking[] {
    return this.getBookingsByStatus("cancelled")
  }

  getBookingById(id: string): HotelBooking | undefined {
    return this.bookings.get(id)
  }

  getBookingStats(): {
    booked: number
    ongoing: number
    completed: number
    cancelled: number
    total: number
    bookedAmount: number
    ongoingAmount: number
    completedAmount: number
    totalAmount: number
    totalCommission: number
  } {
    const all = this.getAllBookings()
    const booked = all.filter((b) => b.status === "booked")
    const ongoing = all.filter((b) => b.status === "ongoing")
    const completed = all.filter((b) => b.status === "completed")
    const cancelled = all.filter((b) => b.status === "cancelled")

    return {
      booked: booked.length,
      ongoing: ongoing.length,
      completed: completed.length,
      cancelled: cancelled.length,
      total: all.length,
      bookedAmount: booked.reduce((sum, b) => sum + b.totalPrice, 0),
      ongoingAmount: ongoing.reduce((sum, b) => sum + b.totalPrice, 0),
      completedAmount: completed.reduce((sum, b) => sum + b.totalPrice, 0),
      totalAmount: [...booked, ...ongoing, ...completed].reduce((sum, b) => sum + b.totalPrice, 0),
      totalCommission: [...booked, ...ongoing, ...completed].reduce((sum, b) => sum + b.commission, 0),
    }
  }
}

export const hotelBookingDataStore = new HotelBookingDataStore()
